package com.platon.crypto;

import com.platon.common.PrivacyAddress;
import org.web3j.crypto.Credentials;
import org.web3j.crypto.ECKeyPair;
import org.web3j.crypto.Keys;
import org.web3j.utils.Numeric;

import java.util.Objects;

/**
 * Credentials wrapper.
 */
public class CredentialsWrapper {

    public Keypair viewKeypair;
    public Keypair spendKeypair;
    private Credentials credentials;
    private PrivacyAddress privacyAddress;

    private CredentialsWrapper(Keypair viewKeypair, Keypair spendKeypair, Credentials credentials) {
        this.viewKeypair = viewKeypair;
        this.spendKeypair = spendKeypair;
        this.credentials = credentials;
        this.privacyAddress = new PrivacyAddress(viewKeypair.getPublicKey(), spendKeypair.getPublicKey());
    }

    public String getPrivacyAddress() {
        return this.privacyAddress.getAddress();
    }

    public String getAddress(long chainId) {
        return this.credentials.getAddress(chainId);
    }

    public PrivacyAddress getPrivacy() {
        return this.privacyAddress;
    }

    public Credentials getCredentials() {
        return credentials;
    }

    public Keypair getViewKeypair() {
        return viewKeypair;
    }

    public Keypair getSpendKeypair() {
        return spendKeypair;
    }

    public static CredentialsWrapper generate(long chainId) {
        Credentials credentials = null;
        try {
            credentials = Credentials.create(Keys.createEcKeyPair());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        Keypair viewKeypair = Keypair.createKeypair(chainId);
        Keypair spendKeypair = Keypair.createKeypair(chainId);
        return new CredentialsWrapper(viewKeypair, spendKeypair, credentials);
    }

    public static CredentialsWrapper randomCreate(ECKeyPair ecKeyPair, long chainId) {
        Credentials credentials = Credentials.create(ecKeyPair);
        Keypair viewKeypair = Keypair.createKeypair(chainId);
        Keypair spendKeypair = Keypair.createKeypair(chainId);
        return new CredentialsWrapper(viewKeypair, spendKeypair, credentials);
    }

    public static CredentialsWrapper create(String viewingKey, String spendingKey, String privateKey, long chainId) {
        Keypair viewingKeypair = Keypair.createKeypair(Numeric.hexStringToByteArray(viewingKey), chainId);
        Keypair spendingKeypair = Keypair.createKeypair(Numeric.hexStringToByteArray(spendingKey), chainId);
        Credentials credentials = Credentials.create(privateKey);
        return new CredentialsWrapper(viewingKeypair, spendingKeypair, credentials);
    }

    public static CredentialsWrapper load(String viewingSk, String viewingPk, String spendingSk, String spendingPk, String privateKey) {
        Keypair viewingKeypair = new Keypair(Numeric.hexStringToByteArray(viewingSk), Numeric.hexStringToByteArray(viewingPk));
        Keypair spendingKeypair = new Keypair(Numeric.hexStringToByteArray(spendingSk), Numeric.hexStringToByteArray(spendingPk));
        Credentials credentials = Credentials.create(privateKey);
        return new CredentialsWrapper(viewingKeypair, spendingKeypair, credentials);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        CredentialsWrapper that = (CredentialsWrapper) o;
        return Objects.equals(viewKeypair, that.viewKeypair) &&
                Objects.equals(spendKeypair, that.spendKeypair) &&
                Objects.equals(credentials, that.credentials);
    }

    @Override
    public int hashCode() {
        return Objects.hash(viewKeypair, spendKeypair, credentials);
    }
}
