package com.platon.storage.db;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class TrackDatabase implements Database {

    private Database db;

    private String DEFAULT_CONTRACT_KEY = "default_contract";
    public Map<String, ContractWrapper> changesMap = new ConcurrentHashMap<>();

    public TrackDatabase(Database db) {
        this.db = db;
    }

    public synchronized void startTrack(String contract) {
        changesMap.remove(contract);
        changesMap.put(contract, new ContractWrapper(contract));
    }

    public synchronized void commitTrack(String contract) {
        ContractWrapper contractWrapper = changesMap.get(contract);
        if (null != contractWrapper) {
            contractWrapper.commit();
        }
    }

    public synchronized void rollbackTrack(String contract) {
        ContractWrapper contractWrapper = changesMap.get(contract);
        if (null != contractWrapper) {
            contractWrapper.rollback();
        }
    }

    public synchronized void put(String contract, byte[] key, byte[] value) {
        ContractWrapper contractWrapper = changesMap.get(contract);
        if (null != contractWrapper) {
            contractWrapper.put(key, value);
        }
    }

    public synchronized byte[] get(String contract, byte[] key) {
        ContractWrapper contractWrapper = changesMap.get(contract);
        if (null != contractWrapper) {
            return contractWrapper.get(key);
        } else {
            return db.get(key);
        }
    }

    /** Delete object (key) from db **/
    public synchronized void delete(String contract, byte[] key) {
        ContractWrapper contractWrapper = changesMap.get(contract);
        if (null != contractWrapper) {
            contractWrapper.delete(key);
        } else {
            db.delete(key);
        }
    }

    @Deprecated
    public void startTrack() {
        changesMap.put(DEFAULT_CONTRACT_KEY, new ContractWrapper(DEFAULT_CONTRACT_KEY));
    }

    @Deprecated
    public void commitTrack() {
        ContractWrapper contractWrapper = changesMap.get(DEFAULT_CONTRACT_KEY);
        if (null != contractWrapper) {
            contractWrapper.commit();
        }
    }

    @Deprecated
    public void rollbackTrack() {
        ContractWrapper contractWrapper = changesMap.get(DEFAULT_CONTRACT_KEY);
        if (null != contractWrapper) {
            contractWrapper.rollback();
        }
    }

    @Override
    public byte[] get(byte[] key) {
        // todo: empty implements
        return get(DEFAULT_CONTRACT_KEY, key);
    }

    @Override
    public void put(byte[] key, byte[] value) {
        // todo: empty implements
        put(DEFAULT_CONTRACT_KEY, key, value);
    }

    @Override
    public void delete(byte[] key) {
        // todo: empty implements
        delete(DEFAULT_CONTRACT_KEY, key);
    }

    @Override
    public void close(){
        db.close();
    }

    public class ContractWrapper {
        private String contract;
        public boolean trackingChanges;
        public Map<ByteArrayWrapper, byte[]> changes;
        public List<ByteArrayWrapper> deletes;

        public ContractWrapper(String contract) {
            this.contract = contract;
            this.changes = new HashMap<>();
            this.deletes = new ArrayList<>();
            this.trackingChanges = true;
        }

        public void commit() {
            for (ByteArrayWrapper key : changes.keySet()) {
                db.put(key.getData(), changes.get(key));
            }
            for (ByteArrayWrapper key : deletes) {
                db.delete(key.getData());
            }
            changes = null;
            trackingChanges = false;
        }

        public void rollback() {
            changes = new HashMap<>();
            deletes = new ArrayList<>();
            changes = null;
            trackingChanges = false;
        }

        public void put(byte[] key, byte[] value) {
            if (trackingChanges) {
                ByteArrayWrapper wKey = new ByteArrayWrapper(key);
                changes.put(wKey, value);
            } else {
                db.put(key, value);
            }
        }

        public byte[] get(byte[] key) {
            if(trackingChanges) {
                ByteArrayWrapper wKey = new ByteArrayWrapper(key);
                if (deletes.contains(wKey)) return null;
                if (changes.get(wKey) != null) return changes.get(wKey);
            }
            return db.get(key);
        }

        /** Delete object (key) from db **/
        public void delete(byte[] key) {
            if (trackingChanges) {
                ByteArrayWrapper wKey = new ByteArrayWrapper(key);
                deletes.add(wKey);
            } else {
                db.delete(key);
            }
        }

        public boolean isTrackingChanges() {
            return trackingChanges;
        }

        public void setTrackingChanges(boolean trackingChanges) {
            this.trackingChanges = trackingChanges;
        }

        public Map<ByteArrayWrapper, byte[]> getChanges() {
            return changes;
        }

        public void setChanges(Map<ByteArrayWrapper, byte[]> changes) {
            this.changes = changes;
        }

        public List<ByteArrayWrapper> getDeletes() {
            return deletes;
        }

        public void setDeletes(List<ByteArrayWrapper> deletes) {
            this.deletes = deletes;
        }
    }
}
