package com.platon.sync;

import com.platon.common.events.response.*;
import com.platon.contract.wrapper.Validator;
import com.platon.contract.wrapper.PrivacyToken;
import com.platon.rlp.datatypes.Uint128;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.web3j.abi.WasmEventValues;
import org.web3j.abi.datatypes.WasmEvent;
import org.web3j.crypto.Hash;
import org.web3j.protocol.core.methods.response.Log;
import org.web3j.tx.WasmContract;
import org.web3j.utils.Numeric;

import java.util.List;

/**
 * Parse the event.
 *
 * @author AgentRJ
 * Created on: 2020-12-18 14:33
 */
public final class EventParser {

    private final static Logger logger = LoggerFactory.getLogger("sync");

    private static WasmEvent burnEvent = PrivacyToken.BURNEVENT_EVENT;
    private static WasmEvent mintEvent = PrivacyToken.MINTEVENT_EVENT;
    private static WasmEvent destroyNoteEvent = PrivacyToken.DESTROYNOTEEVENT_EVENT;
    private static WasmEvent createNoteEvent = PrivacyToken.CREATENOTEEVENT_EVENT;
    private static WasmEvent createDetailNoteEvent = Validator.CREATENOTEDETAILEVENT_EVENT;
    private static WasmEvent metadataEvent = PrivacyToken.METADATAEVENT_EVENT;

    public static void processMintEvent(Log log, PrivacyEventMessage privacyEventMessage, long chainId) {
        final WasmEventValues eventValues = WasmContract.staticExtractEventParameters(mintEvent, log, chainId);
        if (eventValues == null) {
            return;
        }
        // mint Event
        if (logger.isDebugEnabled()) {
            logger.debug("start process mint event, txHash: {}", log.getTransactionHash());
        }
        byte[] mintHash = (byte[]) eventValues.getNonIndexedValues().get(0);
        Uint128 totalMint = (Uint128) eventValues.getNonIndexedValues().get(1);
        MintResponse response = new MintResponse();
        response.setBlockNumber(log.getBlockNumber());
        response.setTxHash(log.getTransactionHash());
        response.setContract(log.getAddress());
        response.setMintHash(Numeric.toHexString(mintHash));
        response.setTotalMint(totalMint.getValue());
        privacyEventMessage.getMintEventResponses().add(response);
    }

    public static void processBurnEvent(Log log, PrivacyEventMessage privacyEventMessage, long chainId) {
        final WasmEventValues eventValues = WasmContract.staticExtractEventParameters(burnEvent, log, chainId);
        if (eventValues == null) {
            return;
        }
        // burn Event
        if (logger.isDebugEnabled()) {
            logger.debug("start process burn event, txHash: {}", log.getTransactionHash());
        }
        byte[] burnHash = (byte[]) eventValues.getNonIndexedValues().get(0);
        Uint128 totalBurn = (Uint128) eventValues.getNonIndexedValues().get(1);
        BurnResponse response = new BurnResponse();
        response.setBlockNumber(log.getBlockNumber());
        response.setTxHash(log.getTransactionHash());
        response.setContract(log.getAddress());
        response.setBurnHash(Numeric.toHexString(burnHash));
        response.setBurnAmount(totalBurn.getValue());
        privacyEventMessage.getBurnEventResponses().add(response);
    }

    public static void processDestroyNoteEvent(Log log, PrivacyEventMessage privacyEventMessage, long chainId) {
        final WasmEventValues eventValues = WasmContract.staticExtractEventParameters(destroyNoteEvent, log, chainId);
        if (eventValues == null) {
            return;
        }
        // destroy Event
        if (logger.isDebugEnabled()) {
            logger.debug("start process destroy event, txHash: {}", log.getTransactionHash());
        }
        String indexHash = eventValues.getIndexedValues().get(1);
        String ownerHash = eventValues.getIndexedValues().get(0);
        byte[] ownerValue = (byte[]) eventValues.getNonIndexedValues().get(0);
        DestroyNoteResponse response = new DestroyNoteResponse();
        response.setBlockNumber(log.getBlockNumber());
        response.setTxHash(log.getTransactionHash());
        response.setContract(log.getAddress());
        response.setNoteHash(indexHash);
        response.setOwnerHash(ownerHash);
        response.setOwnerValue(Numeric.toHexString(ownerValue));
        privacyEventMessage.getDestroyNoteEventResponses().add(response);
    }

    public static CreateNoteResponse processCreateNoteEvent(Log log, PrivacyEventMessage privacyEventMessage, long chainId) {
        final WasmEventValues eventValues = WasmContract.staticExtractEventParameters(createNoteEvent, log, chainId);
        if (eventValues == null) {
            return null;
        }
        // create Event, 根据交易哈希查询该交易下的回执，再次进行一次
        if (logger.isDebugEnabled()) {
            logger.debug("start process createNote event, txHash: {}", log.getTransactionHash());
        }
        String noteHash = eventValues.getIndexedValues().get(1);
        String ownerHash = eventValues.getIndexedValues().get(0);
        byte[] ownerValue = (byte[]) eventValues.getNonIndexedValues().get(0);
        byte[] ownerValueHash = Hash.sha3(ownerValue);
        CreateNoteResponse response = new CreateNoteResponse();
        response.setBlockNumber(log.getBlockNumber());
        response.setTxHash(log.getTransactionHash());
        response.setContract(log.getAddress());
        response.setNoteHash(noteHash);
        response.setOwnerHash(ownerHash);
        response.setOwnerValue(Numeric.toHexString(ownerValue));
        response.setOwnerValueHash(Numeric.toHexString(ownerValueHash));
        privacyEventMessage.getCreateNoteEventResponses().add(response);
        return response;
        // createNote 不能作为新创建的票据事件解析，需要进一步的获取当前交易对应的CreateDetailNoteEvent的数据，
        // 根据相同的交易哈希找到对应的Log数据；
        //processCreateDetailNoteEvent(log.getTransactionHash(), privacyEventMessage, response.getContract(), noteHash);
    }

    public static void processCreateDetailNoteEvent(List<Log> logs, PrivacyEventMessage privacyEventMessage,
                                                    String privacyContractAddress, String noteHash,
                                                    long chainId) {
        if (null == logs) {
            return;
        }
        for (int i = 0; i < logs.size(); i++) {
            Log detailLog = logs.get(i);
            WasmEventValues eventValues = WasmContract.staticExtractEventParameters(createDetailNoteEvent, detailLog, chainId);
            if (null == eventValues) {
                continue;
            }
            // note hash
            String detailNoteHash = eventValues.getIndexedValues().get(0);
            if (!noteHash.equalsIgnoreCase(detailNoteHash)) {
                continue;
            }

            // 票据所有者加密数据，解码获取详细
            byte[] ownerValue = (byte[]) eventValues.getNonIndexedValues().get(0);

            // 票据的加密信息，解码获取详细
            byte[] cipherValue = (byte[]) eventValues.getNonIndexedValues().get(1);

            CreateDetailNoteResponse response = new CreateDetailNoteResponse();
            response.setBlockNumber(detailLog.getBlockNumber());
            response.setTxHash(detailLog.getTransactionHash());
            response.setContract(privacyContractAddress);
            response.setNoteHash(detailNoteHash);
            response.setOwnerValue(Numeric.toHexString(ownerValue));
            response.setCipherValue(Numeric.toHexString(cipherValue));
            privacyEventMessage.getCreateNoteDetailEventResponses().add(response);
            break;
        }
    }

    public static void processMetadataEvent(Log log, PrivacyEventMessage privacyEventMessage, long chainId) {
        final WasmEventValues eventValues = WasmContract.staticExtractEventParameters(metadataEvent, log, chainId);
        if (eventValues == null) {
            return;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("start process metaData event, txHash: {}", log.getTransactionHash());
        }
        String noteHash = eventValues.getIndexedValues().get(0);
        byte[] metaData = (byte[]) eventValues.getNonIndexedValues().get(0);
        MetaDataResponse response = new MetaDataResponse(noteHash, metaData);
        response.setBlockNumber(log.getBlockNumber());
        response.setTxHash(log.getTransactionHash());
        response.setContract(log.getAddress());
        privacyEventMessage.getMetaDataResponses().add(response);
    }

}
