package com.platon.sync;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Comparator;
import java.util.List;
import java.util.Queue;
import java.util.Timer;
import java.util.concurrent.PriorityBlockingQueue;

/**
 * @author AgentRJ
 * Created on: 2020-12-04 17:06
 */
public class EventQueue {

    private static final Logger logger = LoggerFactory.getLogger("sync");

    /**
     * Queue with event to be validated and added to be stored
     */
    private Queue<PrivacyEventMessage> eventReceivedQueue = new PriorityBlockingQueue<>(11, new Comparator<PrivacyEventMessage>() {
        @Override
        public int compare(PrivacyEventMessage o1, PrivacyEventMessage o2) {
            return o1.getBlockNumber().compareTo(o2.getBlockNumber());
        }
    });

    /**
     * Last blockNumber in the queue to be processed
     */
    private long blockNumber;

    private Timer timer = new Timer("EventQueueTimer");

    public EventQueue() {
    }

    public PrivacyEventMessage pollEvent() {
        if (eventReceivedQueue.isEmpty()) {
            return null;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("EventQueue size: {}", eventReceivedQueue.size());
        }
        PrivacyEventMessage privacyEventMessage = eventReceivedQueue.poll();
        logger.info("Processing event index: {}", privacyEventMessage.getBlockNumber());
        return privacyEventMessage;
    }

    /**
     * 批量添加事件响应报文到队列
     *
     * @param messageList
     */
    public void addEventMessages(List<PrivacyEventMessage> messageList) {
        PrivacyEventMessage lastReceiveEvent = messageList.get(0);
        if (lastReceiveEvent.getBlockNumber().longValue() != blockNumber + 1) {
            logger.error("Event download out of sync: lastEvent.index: [{}], receivedEvent.index: [{}]",
                    blockNumber, lastReceiveEvent.getBlockNumber());
            return;
        }
        eventReceivedQueue.addAll(messageList);
        blockNumber = messageList.get(messageList.size() - 1).getBlockNumber().longValue();
        if (logger.isDebugEnabled()) {
            logger.debug("Events waiting to be proceed: queue.size: [{}] lastBlock.number: [{}]",
                    eventReceivedQueue.size(), blockNumber);
        }
    }

    /**
     * 添加一个块的事件结果到队列中
     *
     * @param eventMessage
     */
    public void addEventMessage(PrivacyEventMessage eventMessage) {
        if (eventMessage.getBlockNumber().longValue() != blockNumber + 1) {
            logger.error("Event download out of sync: lastEvent.index: [{}], receivedEvent.index: [{}]",
                    blockNumber, eventMessage.getBlockNumber());
            return;
        }
        eventReceivedQueue.add(eventMessage);
        blockNumber = eventMessage.getBlockNumber().longValue();
        if (logger.isDebugEnabled()) {
            logger.debug("Events waiting to be proceed: queue.size: [{}] lastBlock.number: [{}]",
                    eventReceivedQueue.size(), blockNumber);
        }
    }
}
