package com.platon.sync;

import com.platon.common.events.response.CreateNoteResponse;
import com.platon.common.exception.EventProcessException;
import com.platon.contract.wrapper.Validator;
import com.platon.contract.wrapper.PrivacyToken;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.web3j.abi.WasmEventEncoder;
import org.web3j.abi.datatypes.WasmEvent;
import org.web3j.protocol.Web3j;
import org.web3j.protocol.core.DefaultBlockParameter;
import org.web3j.protocol.core.DefaultBlockParameterNumber;
import org.web3j.protocol.core.methods.request.PlatonFilter;
import org.web3j.protocol.core.methods.response.Log;
import org.web3j.protocol.core.methods.response.TransactionReceipt;
import rx.Observer;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * 事件采集任务，输入指定的块高区间，获取当前区间内的所有回执数据
 * 1、采集到的数据存入消息队列中；
 */
public class WorkerThread implements Callable<List<PrivacyEventMessage>> {

    private final static Logger logger = LoggerFactory.getLogger("sync");

    // Events that need to be filtered.
    private WasmEvent burnEvent = PrivacyToken.BURNEVENT_EVENT;
    private WasmEvent mintEvent = PrivacyToken.MINTEVENT_EVENT;
    private WasmEvent destroyNoteEvent = PrivacyToken.DESTROYNOTEEVENT_EVENT;
    private WasmEvent createNoteEvent = PrivacyToken.CREATENOTEEVENT_EVENT;
    private WasmEvent createDetailNoteEvent = Validator.CREATENOTEDETAILEVENT_EVENT;
    private WasmEvent metadataEvent = PrivacyToken.METADATAEVENT_EVENT;

    private Web3j web3j;
    private long startBn;
    private long endBn;
    private List<String> contracts;
    private long chainId;

    private DefaultBlockParameter startBlock;
    private DefaultBlockParameter endBlock;

    public WorkerThread(Web3j web3j, List<String> contract, long startBn, long endBn, long chainId) {
        this.web3j = web3j;
        this.startBn = startBn;
        this.endBn = endBn;
        this.contracts = contract;
        this.chainId = chainId;

        this.startBlock = new DefaultBlockParameterNumber(BigInteger.valueOf(startBn));
        this.endBlock = new DefaultBlockParameterNumber(BigInteger.valueOf(endBn));
    }

    @Override
    public List<PrivacyEventMessage> call() throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("{} start", Thread.currentThread().getName());
        }
        List<PrivacyEventMessage> privacyEventMessageList = new ArrayList<>();
        PlatonFilter platonFilter = new PlatonFilter(startBlock, endBlock, contracts);
        platonFilter.addOptionalTopics(WasmEventEncoder.encode(burnEvent),
                WasmEventEncoder.encode(mintEvent),
                WasmEventEncoder.encode(destroyNoteEvent),
                WasmEventEncoder.encode(createNoteEvent),
                WasmEventEncoder.encode(metadataEvent));

        final CountDownLatch latch = new CountDownLatch(1);

        // 执行过滤计划
        final Throwable[] exception = new Throwable[2];
        web3j.platonLogObservable(platonFilter).timeout(5, TimeUnit.SECONDS).subscribe(new Observer<Log>() {

            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
                exception[0] = e;
                latch.countDown();
            }

            @Override
            public void onNext(Log log) {
                try {
                    PrivacyEventMessage privacyEventMessage = new PrivacyEventMessage();
                    privacyEventMessage.setContract(log.getAddress());
                    privacyEventMessage.setBlockHash(log.getBlockHash());
                    privacyEventMessage.setBlockNumber(log.getBlockNumber());

                    EventParser.processBurnEvent(log, privacyEventMessage, chainId);
                    EventParser.processMintEvent(log, privacyEventMessage, chainId);
                    EventParser.processDestroyNoteEvent(log, privacyEventMessage, chainId);
                    EventParser.processMetadataEvent(log, privacyEventMessage, chainId);

                    CreateNoteResponse createNoteResponse = EventParser.processCreateNoteEvent(log, privacyEventMessage, chainId);
                    if (null != createNoteResponse) {
                        // parse create detail event
                        Optional<TransactionReceipt> receiptOptional = web3j.platonGetTransactionReceipt(log.getTransactionHash()).send().getTransactionReceipt();
                        if (receiptOptional.isPresent()) {
                            EventParser.processCreateDetailNoteEvent(receiptOptional.get().getLogs(), privacyEventMessage,
                                    createNoteResponse.getContract(), createNoteResponse.getNoteHash(), chainId);
                        }
                    }
                    privacyEventMessageList.add(privacyEventMessage);
                } catch (Exception e) {
                    e.printStackTrace();
                    exception[1] = new EventProcessException(e);
                } finally {
                    latch.countDown();
                }
            }
        });

        try {
            latch.await();
        } finally {
        }

        // execute done.
        if (exception[0] instanceof TimeoutException) {
            if (logger.isDebugEnabled()) {
                logger.debug("process event timeout...");
            }
            //throw new TimeoutException("process event timeout");
        }
        if (exception[1] instanceof EventProcessException) {
            throw new EventProcessException(exception[1]);
        }
        if (logger.isDebugEnabled()) {
            logger.debug("{} ending...", Thread.currentThread().getName());
        }
        return privacyEventMessageList;
    }


}
